package gov.va.med.mhv.vitals.web.controller;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Set;

import javax.faces.application.FacesMessage;
import javax.faces.bean.ManagedBean;
import javax.faces.context.FacesContext;
import javax.faces.event.ComponentSystemEvent;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.jfree.data.time.Day;
import org.jfree.data.time.TimeSeries;
import org.jfree.data.time.TimeSeriesCollection;
import org.jfree.data.xy.XYDataset;
import org.primefaces.component.datatable.DataTable;
import org.primefaces.event.data.SortEvent;
import org.primefaces.model.StreamedContent;
import org.springframework.context.annotation.Scope;
import org.springframework.stereotype.Component;

import gov.va.med.mhv.common.api.exception.MHVException;
import gov.va.med.mhv.vitals.dto.LipidsReadingDTO;
import gov.va.med.mhv.vitals.enums.PeriodEnumeration;
import gov.va.med.mhv.vitals.util.CommonUtility;
import gov.va.med.mhv.vitals.web.util.WebUtility;

@ManagedBean
@Component
@Scope("session")
public class LipidsController extends AbstractController {

	private static final long serialVersionUID = 134538500296925325L;
	private static Logger log = LogManager.getLogger(LipidsController.class);
	private List<LipidsReadingDTO> lpreadings = new ArrayList<LipidsReadingDTO>();
	private LipidsReadingDTO selectedLPReading = new LipidsReadingDTO();
	private LipidsReadingDTO newLPReading = new LipidsReadingDTO();
	private static final String TITLE = "Cholesterol (Lipids Profile)";
	private static final String XAXIS = "Date";
	private static final String YAXIS = "mg/dL";
	private static final String SERIES1 = "Total";
	private static final String SERIES2 = "HDL";
	private static final String SERIES3 = "LDL";
	private static final String SERIES4 = "TGL";

	public void init(ComponentSystemEvent event) {
		findUser();
		userprofileId = getUserProfileId();
		DataTable lpTable = (DataTable) FacesContext.getCurrentInstance().getViewRoot().findComponent("lipidsForm:lpList");
		if (!FacesContext.getCurrentInstance().isPostback()) {
			resetMessages();
			if (userprofileId != null) {
				lpreadings = findLPReadings(userprofileId);
			}
			setRowsPerPage(10);
			setPeriod(PeriodEnumeration.ONE_MONTH.getDescription());
		}
		else{
			if(sortColumn != null && sortBy != null){
				lpTable.setValueExpression("sortBy", sortColumn);
				lpTable.setSortOrder(sortBy);
			} 
		}
	}
	
	public void onSort(SortEvent event){
		sortColumn=event.getSortColumn().getValueExpression("sortBy");
		sortBy=event.isAscending()?"ascending":"descending";
	}

	private List<LipidsReadingDTO> findLPReadings(Long userprofileId) {
		List<LipidsReadingDTO> dtoList = null;

		try {

			dtoList = this.vitalSignsService.getLipidReadingsForUser(userprofileId);

		} catch (Exception e) {
			log.error("Error in Find Lipid readings:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return dtoList;
	}

	public String showDetail(LipidsReadingDTO lipidsReadingDTO) {
		resetMessages();
		selectedLPReading = lipidsReadingDTO;
		setHourMinute(selectedLPReading);
		return "lipids";
	}
	
	public String editDisplay(LipidsReadingDTO lipidsReadingDTO) {
		resetMessages();
		selectedLPReading = lipidsReadingDTO;
		setHourMinute(selectedLPReading);
		return "editLPDisplay";
	}
	
	public String resetEditDisplay(){
		return "editLPDisplay";
	}

	public String deleteDisplay(LipidsReadingDTO lipidsReadingDTO) {
		resetMessages();
		selectedLPReading = lipidsReadingDTO;
		setHourMinute(selectedLPReading);
		setDeleteOrigin("tableView");
		return "deleteLPDisplay";
	}

	public String deleteRecordDisplay() {
		resetMessages();
		setDeleteOrigin(null);
		return "deleteLPDisplay";
	}

	private void setHourMinute(LipidsReadingDTO selectedLPReading) {
		String dateTime = WebUtility.dateToString(selectedLPReading.getReading(), HOUR_MINUTE);
		if (!dateTime.equals(DEFAULT_DATETIME)) {
			selectedLPReading.setHour(WebUtility.dateToString(selectedLPReading.getReading(), HOURS));
			selectedLPReading.setMinute(WebUtility.dateToString(selectedLPReading.getReading(), MINUTES));
		}

	}

	public String addDisplay() {
		resetMessages();
		newLPReading = new LipidsReadingDTO();
		newLPReading.setReading(new Date());
		return "addLPDisplay";
	}

	public String dashboardAddDisplay() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			lpreadings = findLPReadings(userprofileId);
		}
		newLPReading = new LipidsReadingDTO();
		newLPReading.setReading(new Date());
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().handleNavigation(context, null,
				"/views/lipidsprofile/addLPDisplay.xhtml");
		return null;
	}
	
	public String dashboardViewMore() {
		resetMessages();
		findUser();
		setRowsPerPage(10);
		userprofileId = getUserProfileId();
		if (userprofileId != null) {
			lpreadings = findLPReadings(userprofileId);
		}
		return "/views/lipidsprofile/lipidsList";
	}

	public String showDahBoardDetail() {
		resetMessages();
		findUser();
		Long userProfileId = getUserProfileId();
		String lpid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("lpid");
		if (lpid != null && userProfileId != null) {
			Long id = Long.valueOf(lpid);
			selectedLPReading = getLPReadingById(userProfileId, id);
		}
		FacesContext context = FacesContext.getCurrentInstance();
		context.getApplication().getNavigationHandler().handleNavigation(context, null,
				"/views/lipidsprofile/lipids.xhtml");
		return null;
	}

	private LipidsReadingDTO getLPReadingById(Long userProfileId, Long id) {
		LipidsReadingDTO dto = null;
		try {

			dto = this.vitalSignsService.getLipidReadingByid(userProfileId, id);

		} catch (Exception e) {
			log.error("Error in Find LP reading By id:", e);
			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return dto;
	}

	public String delete() {
		String outcome = null;
		String lpid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("lpid");
		Long userProfileId = getUserProfileId();

		try {
			if (lpid != null && userProfileId != null) {
				Long id = Long.valueOf(lpid);

				this.vitalSignsService.deleteLipidsReading(userProfileId, id);
				lpreadings = findLPReadings(userprofileId);
				deleteMessage = true;
				outcome = "lipidsList";
			}
		} catch (Exception e) {
			log.error("Failed to Delete Lipids record", e);

			FacesContext.getCurrentInstance().addMessage(null,
					new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
		}

		return outcome;
	}

	public String save() {
		String outcome = null;
		try {
			if (saveLPReading().getLipidsId() != null) {
				outcome = "lipidsList";
				lpreadings = findLPReadings(userprofileId);
				saveMessage = true;
			}
		} catch (MHVException e) {
			log.error("Failed to Save Lipids record", e);

			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}

	public LipidsReadingDTO saveLPReading() throws MHVException {
		resetMessages();
		String lpid = FacesContext.getCurrentInstance().getExternalContext().getRequestParameterMap().get("lpid");
		Long id = null;
		LipidsReadingDTO lpreading = null;
		if (lpid != null) {
			id = Long.valueOf(lpid);
			lpreading = getSelectedLPReading();
		} else {
			lpreading = getNewLPReading();
		}

		lpreading = prepareLPReading(id, lpreading);
		Long lpId = this.vitalSignsService.saveLipidsReading(lpreading);
		lpreading.setLipidsId(lpId);
		return lpreading;

	}

	private LipidsReadingDTO prepareLPReading(Long id, LipidsReadingDTO lpreading) {
		Date dateTime = null;
		if (lpreading.getReading() != null) {
			if (lpreading.getHour() == null && lpreading.getMinute() == null) {
				dateTime = WebUtility.getDaTeTime(lpreading.getReading(), DEFAULT_HOUR, DEFAULT_MINUTE);
			} else {
				dateTime = WebUtility.getDaTeTime(lpreading.getReading(), lpreading.getHour(), lpreading.getMinute());
			}
		}
		lpreading.setReading(dateTime);
		lpreading.setUserprofileId(getUserProfileId());
		return lpreading;
	}

	public String saveAndAdd() {
		String outcome = null;
		try {
			if (saveLPReading().getLipidsId() != null) {
				outcome = "addLPDisplay";
				lpreadings = findLPReadings(userprofileId);
				saveAndAddMessage = true;
				newLPReading = new LipidsReadingDTO();
				newLPReading.setReading(new Date());
			}

		}catch (MHVException e) {
			log.error("Failed to Save Lipids record", e);

			if (e.getValidationErrorMessages() != null) {
				addValidationMessages(e.getValidationErrorMessages());
			} else {
				FacesContext.getCurrentInstance().addMessage(null,
						new FacesMessage(FacesMessage.SEVERITY_ERROR, ERR_PRCS_RQST, ERR_PRCS_RQST));
			}
		}
		return outcome;
	}

	public String updateLineModel() {
		return "lipidsGraph";
	}

	public StreamedContent getJchart() {
		StreamedContent content = null;
		try {
			content = createChart(TITLE, XAXIS, YAXIS);
		} catch (Exception e) {
			log.error("Failed to create Chart", e);
		}
		return content;
	}

	protected XYDataset createDataset(Set<String> uniqueYears) {
		TimeSeriesCollection dataset = new TimeSeriesCollection();
		TimeSeries series1 = new TimeSeries(SERIES1);
		TimeSeries series2 = new TimeSeries(SERIES2);
		TimeSeries series3 = new TimeSeries(SERIES3);
		TimeSeries series4 = new TimeSeries(SERIES4);
		String period = getPeriod();
		Date startDate = null;
		List<LipidsReadingDTO> graphList;
		if (!period.equals(PeriodEnumeration.ALL_DATA.getDescription())) {
			startDate = getStartGraphDate(period);
		}
		if (startDate != null) {
			graphList = getGraphlist(lpreadings, startDate);
		} else {
			graphList = lpreadings;
		}
		for (LipidsReadingDTO lpreading : graphList) {
			series1.addOrUpdate(new Day(lpreading.getReading()), lpreading.getTotal());
			series2.addOrUpdate(new Day(lpreading.getReading()), lpreading.getHdl());
			series3.addOrUpdate(new Day(lpreading.getReading()), lpreading.getLdl());
			series4.addOrUpdate(new Day(lpreading.getReading()), lpreading.getTriglycerides());
			uniqueYears
					.add(CommonUtility.dateToString(lpreading.getReading(), YYYY_MM_DD_HHMMSS_FORMAT).substring(0, 4));
		}
		dataset.addSeries(series1);
		dataset.addSeries(series2);
		dataset.addSeries(series3);
		dataset.addSeries(series4);
		return dataset;
	}

	private List<LipidsReadingDTO> getGraphlist(List<LipidsReadingDTO> lpreadings, Date startDate) {
		List<LipidsReadingDTO> graphList = new ArrayList<LipidsReadingDTO>();
		Integer startdateValue = Integer.valueOf(CommonUtility.dateToString(startDate, YYYYMMDD_FORMAT));
		for (LipidsReadingDTO lpreading : lpreadings) {
			Integer readingValue = Integer.valueOf(CommonUtility.dateToString(lpreading.getReading(), YYYYMMDD_FORMAT));
			if (readingValue >= startdateValue) {
				graphList.add(lpreading);
			}
		}
		return graphList;
	}

	
	public String printerFriendlySummary(){
		resetMessages();
		return "printLPSummary";
	}


	public List<LipidsReadingDTO> getLpreadings() {
		return lpreadings;
	}

	public void setLpreadings(List<LipidsReadingDTO> lpreadings) {
		this.lpreadings = lpreadings;
	}

	public LipidsReadingDTO getSelectedLPReading() {
		return selectedLPReading;
	}

	public void setSelectedLPReading(LipidsReadingDTO selectedLPReading) {
		this.selectedLPReading = selectedLPReading;
	}

	public LipidsReadingDTO getNewLPReading() {
		return newLPReading;
	}

	public void setNewLPReading(LipidsReadingDTO newLPReading) {
		this.newLPReading = newLPReading;
	}

}
